"""Different pickable items."""
from enum import Enum

from dm_control import mjcf
from mojo import Mojo
from mujoco_utils import mjcf_utils

from bigym.const import ASSETS_PATH
from bigym.envs.props.prop import Prop

BOX_MODEL_PATH = ASSETS_PATH / "props/box/box.xml"


class Box(Prop):
    """Box."""

    def __init__(self, mojo: Mojo, kinematic: bool = True):
        """Init."""
        super().__init__(mojo, BOX_MODEL_PATH, kinematic, True)


CUBE_MODEL_PATH = ASSETS_PATH / "props/cube/cube.xml"


class Cube(Prop):
    """Cube."""

    def __init__(self, mojo: Mojo, kinematic: bool = True):
        """Init."""
        super().__init__(mojo, CUBE_MODEL_PATH, kinematic, True)


SANDWICH_MODEL_PATH = ASSETS_PATH / "props/sandwich/sandwich.xml"
SANDWICH_TOASTED_MODEL_PATH = ASSETS_PATH / "props/sandwich/sandwich_toasted.xml"


class Sandwich(Prop):
    """Sandwich."""

    _COLLIDER = "collider"
    _COLLIDER_ROUNDED = "collider_rounded"

    def __init__(
        self,
        mojo: Mojo,
        kinematic: bool = True,
        toasted: bool = False,
        rounded: bool = False,
    ):
        """Init."""
        self._rounded = rounded
        model_path = SANDWICH_TOASTED_MODEL_PATH if toasted else SANDWICH_MODEL_PATH
        super().__init__(mojo, model_path, kinematic, True)

    def _on_loaded(self, model: mjcf.RootElement):
        remove_collider = self._COLLIDER if self._rounded else self._COLLIDER_ROUNDED
        collider = mjcf_utils.safe_find(model, "geom", remove_collider)
        collider.remove()


class GroceryCategory(Enum):
    """Grocery item category."""

    WINE = ASSETS_PATH / "props/groceries/wine/wine.xml"
    DETERGENT = ASSETS_PATH / "props/groceries/detergent/detergent.xml"
    SOAP = ASSETS_PATH / "props/groceries/soap/soap.xml"
    BEER = ASSETS_PATH / "props/groceries/beer.xml"
    CEREALS = ASSETS_PATH / "props/groceries/cereal.xml"
    CRISPS = ASSETS_PATH / "props/groceries/crisps.xml"
    KETCHUP = ASSETS_PATH / "props/groceries/ketchup.xml"
    MUSTARD = ASSETS_PATH / "props/groceries/mustard.xml"
    SODA = ASSETS_PATH / "props/groceries/soda.xml"


class GroceryItem(Prop):
    """Grocery item."""

    def __init__(
        self, mojo: Mojo, item_category: GroceryCategory, kinematic: bool = True
    ):
        """Init."""
        super().__init__(mojo, item_category.value, kinematic, True)
